<?php
require_once 'config.php';
requireLogin();

$conn = getDB();
$currentUser = getCurrentUser();
$user_id = $currentUser['id'];

// Get user statistics
$stats = [
    'quiz_count' => 0,
    'total_progress' => 0,
    'known_words' => 0,
    'learning_words' => 0,
    'average_score' => 0
];

$stmt = $conn->prepare("SELECT COUNT(*) as total FROM quiz_results WHERE user_id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = fetchStmtAssoc($stmt);
$stats['quiz_count'] = $result['total'] ?? 0;
$stmt->close();

$stmt = $conn->prepare("SELECT COUNT(*) as total FROM user_progress WHERE user_id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = fetchStmtAssoc($stmt);
$stats['total_progress'] = $result['total'] ?? 0;
$stmt->close();

$stmt = $conn->prepare("SELECT COUNT(*) as total FROM user_progress WHERE user_id = ? AND known_status = 'known'");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = fetchStmtAssoc($stmt);
$stats['known_words'] = $result['total'] ?? 0;
$stmt->close();

$stmt = $conn->prepare("SELECT COUNT(*) as total FROM user_progress WHERE user_id = ? AND known_status = 'learning'");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = fetchStmtAssoc($stmt);
$stats['learning_words'] = $result['total'] ?? 0;
$stmt->close();

$stmt = $conn->prepare("SELECT AVG(score) as avg_score FROM quiz_results WHERE user_id = ?");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$avgResult = fetchStmtAssoc($stmt);
$stats['average_score'] = round($avgResult['avg_score'] ?? 0, 1);
$stmt->close();

// Get recent quiz results
$stmt = $conn->prepare("SELECT * FROM quiz_results WHERE user_id = ? ORDER BY completed_at DESC LIMIT 10");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$recent_quizzes = fetchAllAssoc($stmt);
?>
<!DOCTYPE html>
<html lang="vi">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Hồ Sơ - Học Tiếng Trung HSK</title>
    <link rel="stylesheet" href="assets/css/style.css">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Noto+Sans+SC:wght@300;400;500;700&family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
</head>
<body>
    <nav class="navbar">
        <div class="container">
            <div class="nav-brand">
                <h1>中文学习</h1>
                <span>HSK Vocabulary Learning</span>
            </div>
            <ul class="nav-menu">
                <li><a href="index.php">Trang Chủ</a></li>
                <li><a href="vocabulary.php">Từ Vựng</a></li>
                <li><a href="flashcard.php">Flashcard</a></li>
                <li><a href="quiz.php">Kiểm Tra</a></li>
                <li class="user-menu">
                    <a href="#" class="user-menu-toggle">
                        <?php echo htmlspecialchars($currentUser['full_name'] ?? $currentUser['username']); ?>
                        <span class="user-menu-arrow">▼</span>
                    </a>
                    <ul class="user-dropdown">
                        <li><a href="profile.php" class="active">Hồ Sơ</a></li>
                        <?php if (isAdmin()): ?>
                            <li><a href="admin/index.php">Admin Panel</a></li>
                        <?php endif; ?>
                        <li><a href="logout.php">Đăng Xuất</a></li>
                    </ul>
                </li>
            </ul>
        </div>
    </nav>

    <main>
        <section class="profile-section">
            <div class="container">
                <h1 class="page-title">Hồ Sơ Của Tôi</h1>
                
                <div class="profile-card">
                    <div class="profile-header">
                        <div class="profile-avatar">
                            <span><?php echo strtoupper(substr($currentUser['username'], 0, 1)); ?></span>
                        </div>
                        <div class="profile-info">
                            <h2><?php echo htmlspecialchars($currentUser['full_name'] ?? $currentUser['username']); ?></h2>
                            <p><?php echo htmlspecialchars($currentUser['email']); ?></p>
                            <?php if ($currentUser['is_admin']): ?>
                                <span class="badge badge-admin">Admin</span>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>

                <div class="stats-grid" style="margin-top: 2rem;">
                    <div class="stat-card stat-primary">
                        <div class="stat-icon">📝</div>
                        <div class="stat-content">
                            <div class="stat-number"><?php echo number_format($stats['quiz_count']); ?></div>
                            <div class="stat-label">Bài Kiểm Tra</div>
                        </div>
                    </div>
                    
                    <div class="stat-card stat-info">
                        <div class="stat-icon">📊</div>
                        <div class="stat-content">
                            <div class="stat-number"><?php echo number_format($stats['total_progress']); ?></div>
                            <div class="stat-label">Tổng Tiến Độ</div>
                        </div>
                    </div>
                    
                    <div class="stat-card stat-success">
                        <div class="stat-icon">✓</div>
                        <div class="stat-content">
                            <div class="stat-number"><?php echo number_format($stats['known_words']); ?></div>
                            <div class="stat-label">Từ Đã Biết</div>
                        </div>
                    </div>
                    
                    <div class="stat-card stat-warning">
                        <div class="stat-icon">📖</div>
                        <div class="stat-content">
                            <div class="stat-number"><?php echo number_format($stats['learning_words']); ?></div>
                            <div class="stat-label">Đang Học</div>
                        </div>
                    </div>
                    
                    <div class="stat-card stat-info">
                        <div class="stat-icon">⭐</div>
                        <div class="stat-content">
                            <div class="stat-number"><?php echo $stats['average_score']; ?>%</div>
                            <div class="stat-label">Điểm Trung Bình</div>
                        </div>
                    </div>
                </div>

                <div class="profile-section-card">
                    <h2>Kết Quả Kiểm Tra Gần Đây</h2>
                    <?php if (empty($recent_quizzes)): ?>
                        <p style="text-align: center; color: var(--text-secondary); padding: 2rem;">Chưa có kết quả kiểm tra nào</p>
                    <?php else: ?>
                        <div class="table-container">
                            <table class="data-table">
                                <thead>
                                    <tr>
                                        <th>HSK Level</th>
                                        <th>Loại</th>
                                        <th>Số Câu</th>
                                        <th>Đúng</th>
                                        <th>Điểm</th>
                                        <th>Ngày Hoàn Thành</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($recent_quizzes as $quiz): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($quiz['hsk_level'] ?? '-'); ?></td>
                                            <td><?php echo htmlspecialchars($quiz['quiz_type'] ?? '-'); ?></td>
                                            <td><?php echo $quiz['total_questions']; ?></td>
                                            <td><?php echo $quiz['correct_answers']; ?></td>
                                            <td><strong><?php echo $quiz['score']; ?>%</strong></td>
                                            <td><?php echo date('d/m/Y H:i', strtotime($quiz['completed_at'])); ?></td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </section>
    </main>

    <footer class="footer">
        <div class="container">
            <p>&copy; 2024 Học Tiếng Trung HSK. Tất cả quyền được bảo lưu.</p>
        </div>
    </footer>

    <style>
        .profile-section {
            padding: 2rem 0;
            flex: 1;
        }
        
        .profile-card {
            background: white;
            padding: 2rem;
            border-radius: 12px;
            box-shadow: var(--shadow);
            margin-bottom: 2rem;
        }
        
        .profile-header {
            display: flex;
            align-items: center;
            gap: 2rem;
        }
        
        .profile-avatar {
            width: 100px;
            height: 100px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-dark) 100%);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 2.5rem;
            font-weight: 700;
        }
        
        .profile-info h2 {
            margin-bottom: 0.5rem;
            color: var(--text-primary);
        }
        
        .profile-info p {
            color: var(--text-secondary);
            margin-bottom: 0.5rem;
        }
        
        .profile-section-card {
            background: white;
            padding: 2rem;
            border-radius: 12px;
            box-shadow: var(--shadow);
            margin-top: 2rem;
        }
        
        .profile-section-card h2 {
            margin-bottom: 1.5rem;
            color: var(--text-primary);
        }
        
        .data-table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .data-table thead {
            background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-dark) 100%);
            color: white;
        }
        
        .data-table th,
        .data-table td {
            padding: 1rem;
            text-align: left;
            border-bottom: 1px solid var(--border-color);
        }
        
        .data-table tbody tr:hover {
            background-color: #f5f5f5;
        }
    </style>
</body>
</html>

