<?php
// Enable error reporting for debugging (remove in production)
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('log_errors', 1);

// Database configuration
define('DB_HOST', 'localhost');
define('DB_USER', 'root');
define('DB_PASS', '');
define('DB_NAME', 'hsk_learning');

// Site configuration
define('SITE_NAME', 'Học Tiếng Trung HSK');
define('SITE_URL', 'http://localhost/hsk-learning');

// Create database connection
function getDB() {
    static $conn = null;
    
    if ($conn === null) {
        try {
            $conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
            
            if ($conn->connect_error) {
                // Try to create database if it doesn't exist
                $conn_temp = new mysqli(DB_HOST, DB_USER, DB_PASS);
                if ($conn_temp->connect_error) {
                    error_log("Database connection failed: " . $conn_temp->connect_error);
                    // Don't die, return null instead for better error handling
                    return null;
                }
                
                $sql = "CREATE DATABASE IF NOT EXISTS " . DB_NAME . " CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci";
                if ($conn_temp->query($sql) === TRUE) {
                    $conn_temp->close();
                    $conn = new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);
                    if ($conn->connect_error) {
                        error_log("Database connection failed after creation: " . $conn->connect_error);
                        return null;
                    }
                } else {
                    error_log("Error creating database: " . $conn_temp->error);
                    $conn_temp->close();
                    return null;
                }
            }
            
            if ($conn && !$conn->set_charset("utf8mb4")) {
                error_log("Error setting charset: " . $conn->error);
            }
            
            // Create tables if they don't exist
            if ($conn) {
                createTables($conn);
            }
            
        } catch (Exception $e) {
            error_log("Database connection error: " . $e->getMessage());
            return null;
        }
    }
    
    return $conn;
}

function createTables($conn) {
    if (!$conn) {
        return;
    }
    
    // Disable foreign key checks temporarily
    $conn->query("SET FOREIGN_KEY_CHECKS = 0");
    
    try {
        // Users table
        $sql = "CREATE TABLE IF NOT EXISTS users (
            id INT AUTO_INCREMENT PRIMARY KEY,
            username VARCHAR(50) UNIQUE NOT NULL,
            email VARCHAR(100) UNIQUE NOT NULL,
            password VARCHAR(255) NOT NULL,
            full_name VARCHAR(100),
            is_admin TINYINT(1) DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            last_login TIMESTAMP NULL,
            status ENUM('active', 'inactive', 'banned') DEFAULT 'active',
            INDEX idx_username (username),
            INDEX idx_email (email)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        if (!$conn->query($sql)) {
            throw new Exception("Error creating users table: " . $conn->error);
        }
        
        // User progress table
        $sql = "CREATE TABLE IF NOT EXISTS user_progress (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id INT NOT NULL,
            word_simplified VARCHAR(50) NOT NULL,
            hsk_level INT NOT NULL,
            known_status ENUM('known', 'unknown', 'learning') DEFAULT 'learning',
            times_reviewed INT DEFAULT 0,
            last_reviewed TIMESTAMP NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            INDEX idx_user_word (user_id, word_simplified),
            INDEX idx_hsk_level (hsk_level)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        if (!$conn->query($sql)) {
            throw new Exception("Error creating user_progress table: " . $conn->error);
        }
        
        // Quiz results table
        $sql = "CREATE TABLE IF NOT EXISTS quiz_results (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id INT NOT NULL,
            hsk_level INT NOT NULL,
            total_questions INT NOT NULL,
            correct_answers INT NOT NULL,
            score INT NOT NULL,
            quiz_type VARCHAR(20),
            completed_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            INDEX idx_user_date (user_id, completed_at),
            INDEX idx_hsk_level (hsk_level)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        if (!$conn->query($sql)) {
            throw new Exception("Error creating quiz_results table: " . $conn->error);
        }
        
        // Exercises table - for practice exercises by HSK level
        $sql = "CREATE TABLE IF NOT EXISTS exercises (
            id INT AUTO_INCREMENT PRIMARY KEY,
            user_id INT NOT NULL,
            hsk_level INT NOT NULL,
            exercise_type VARCHAR(50) NOT NULL,
            word_simplified VARCHAR(50),
            user_answer TEXT,
            correct_answer TEXT,
            is_correct TINYINT(1) DEFAULT 0,
            completed_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
            INDEX idx_user_level (user_id, hsk_level),
            INDEX idx_exercise_type (exercise_type)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        if (!$conn->query($sql)) {
            throw new Exception("Error creating exercises table: " . $conn->error);
        }
        
        // Re-enable foreign key checks
        $conn->query("SET FOREIGN_KEY_CHECKS = 1");
        
        // Create default admin user if not exists
        $adminCheck = $conn->query("SELECT id FROM users WHERE username = 'admin'");
        if ($adminCheck && $adminCheck->num_rows == 0) {
            $adminPassword = password_hash('admin123', PASSWORD_DEFAULT);
            $stmt = $conn->prepare("INSERT INTO users (username, email, password, full_name, is_admin) VALUES (?, ?, ?, ?, ?)");
            if ($stmt) {
                $username = 'admin';
                $email = 'admin@hsklearning.com';
                $full_name = 'Administrator';
                $is_admin = 1;
                $stmt->bind_param("ssssi", $username, $email, $adminPassword, $full_name, $is_admin);
                if (!$stmt->execute()) {
                    error_log("Error creating admin user: " . $stmt->error);
                }
                $stmt->close();
            }
        }
        
    } catch (Exception $e) {
        $conn->query("SET FOREIGN_KEY_CHECKS = 1");
        error_log("Database table creation error: " . $e->getMessage());
    }
}

// Session management
function startSession() {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
}

function isLoggedIn() {
    startSession();
    return isset($_SESSION['user_id']);
}

function isAdmin() {
    startSession();
    return isset($_SESSION['is_admin']) && $_SESSION['is_admin'] == 1;
}

function requireLogin() {
    if (!isLoggedIn()) {
        // Save current URL to redirect after login
        $_SESSION['redirect_after_login'] = $_SERVER['REQUEST_URI'];
        header('Location: login.php');
        exit;
    }
}

function requireAdmin() {
    requireLogin();
    if (!isAdmin()) {
        header('Location: index.php');
        exit;
    }
}

// Helper function to get result from prepared statement (compatible with both mysqlnd and libmysql)
function getStmtResult($stmt) {
    if (method_exists($stmt, 'get_result')) {
        // MySQLnd driver
        return $stmt->get_result();
    } else {
        // libmysql driver - use store_result
        $stmt->store_result();
        return $stmt;
    }
}

// Helper function to get num_rows (compatible with both drivers)
function getStmtNumRows($stmt) {
    if (method_exists($stmt, 'get_result')) {
        $result = $stmt->get_result();
        return $result->num_rows;
    } else {
        $stmt->store_result();
        return $stmt->num_rows;
    }
}

// Helper function to fetch associative array from statement
function fetchStmtAssoc($stmt) {
    if (method_exists($stmt, 'get_result')) {
        // MySQLnd driver
        $result = $stmt->get_result();
        return $result->fetch_assoc();
    } else {
        // libmysql driver - use bind_result
        $stmt->store_result();
        $meta = $stmt->result_metadata();
        if (!$meta) {
            return null;
        }
        
        $params = [];
        $row = [];
        $fields = [];
        while ($field = $meta->fetch_field()) {
            $fields[] = $field->name;
            $params[] = &$row[$field->name];
        }
        call_user_func_array([$stmt, 'bind_result'], $params);
        
        if ($stmt->fetch()) {
            $meta->free();
            // Create a copy of the row array to avoid reference issues
            $result = [];
            foreach ($fields as $field) {
                $result[$field] = $row[$field];
            }
            return $result;
        }
        $meta->free();
        return null;
    }
}

// Helper function to fetch all rows from mysqli_result or statement (compatible with both drivers)
function fetchAllAssoc($result) {
    if (is_a($result, 'mysqli_result')) {
        // Direct mysqli_result object
        if (method_exists($result, 'fetch_all')) {
            // MySQLnd driver
            return $result->fetch_all(MYSQLI_ASSOC);
        } else {
            // libmysql driver - fetch manually
            $rows = [];
            while ($row = $result->fetch_assoc()) {
                $rows[] = $row;
            }
            return $rows;
        }
    } else {
        // Prepared statement
        if (method_exists($result, 'get_result')) {
            // MySQLnd driver
            $mysqli_result = $result->get_result();
            if (method_exists($mysqli_result, 'fetch_all')) {
                return $mysqli_result->fetch_all(MYSQLI_ASSOC);
            } else {
                $rows = [];
                while ($row = $mysqli_result->fetch_assoc()) {
                    $rows[] = $row;
                }
                return $rows;
            }
        } else {
            // libmysql driver - use bind_result
            $result->store_result();
            $meta = $result->result_metadata();
            if (!$meta) {
                return [];
            }
            
            $rows = [];
            $params = [];
            $row = [];
            $fields = [];
            
            while ($field = $meta->fetch_field()) {
                $fields[] = $field->name;
                $params[] = &$row[$field->name];
            }
            call_user_func_array([$result, 'bind_result'], $params);
            
            while ($result->fetch()) {
                // Create a copy of the row array to avoid reference issues
                $rowCopy = [];
                foreach ($fields as $field) {
                    $rowCopy[$field] = $row[$field];
                }
                $rows[] = $rowCopy;
            }
            
            $meta->free();
            return $rows;
        }
    }
}

function getCurrentUser() {
    startSession();
    if (!isLoggedIn()) {
        return null;
    }
    
    try {
        $conn = getDB();
        if (!$conn) {
            error_log("Database connection failed in getCurrentUser()");
            return null;
        }
        
        $stmt = $conn->prepare("SELECT id, username, email, full_name, is_admin FROM users WHERE id = ?");
        if (!$stmt) {
            error_log("Prepare failed in getCurrentUser(): " . $conn->error);
            return null;
        }
        
        $stmt->bind_param("i", $_SESSION['user_id']);
        if (!$stmt->execute()) {
            error_log("Execute failed in getCurrentUser(): " . $stmt->error);
            $stmt->close();
            return null;
        }
        
        $user = fetchStmtAssoc($stmt);
        $stmt->close();
        
        return $user;
    } catch (Exception $e) {
        error_log("Error in getCurrentUser(): " . $e->getMessage());
        return null;
    }
}

function getUserProgress($user_id, $hsk_level = null) {
    $conn = getDB();
    if (!$conn) {
        return ['total' => 0, 'known' => 0, 'learning' => 0];
    }
    
    try {
        if ($hsk_level) {
            $stmt = $conn->prepare("SELECT COUNT(*) as total, 
                                    SUM(CASE WHEN known_status = 'known' THEN 1 ELSE 0 END) as known,
                                    SUM(CASE WHEN known_status = 'learning' THEN 1 ELSE 0 END) as learning
                                    FROM user_progress WHERE user_id = ? AND hsk_level = ?");
            if ($stmt) {
                $stmt->bind_param("ii", $user_id, $hsk_level);
                if ($stmt->execute()) {
                    $result = fetchStmtAssoc($stmt);
                    $stmt->close();
                    return $result ? $result : ['total' => 0, 'known' => 0, 'learning' => 0];
                }
                $stmt->close();
            }
        } else {
            $stmt = $conn->prepare("SELECT COUNT(*) as total, 
                                    SUM(CASE WHEN known_status = 'known' THEN 1 ELSE 0 END) as known,
                                    SUM(CASE WHEN known_status = 'learning' THEN 1 ELSE 0 END) as learning
                                    FROM user_progress WHERE user_id = ?");
            if ($stmt) {
                $stmt->bind_param("i", $user_id);
                if ($stmt->execute()) {
                    $result = fetchStmtAssoc($stmt);
                    $stmt->close();
                    return $result ? $result : ['total' => 0, 'known' => 0, 'learning' => 0];
                }
                $stmt->close();
            }
        }
    } catch (Exception $e) {
        error_log("Error in getUserProgress(): " . $e->getMessage());
    }
    
    return ['total' => 0, 'known' => 0, 'learning' => 0];
}
?>
