// Vocabulary page JavaScript
let currentPage = 1;
let currentLevel = window.initialLevel || 0;
let currentSearch = '';
const itemsPerPage = 50;

document.addEventListener('DOMContentLoaded', function() {
    // Level filter buttons
    document.querySelectorAll('.level-btn').forEach(btn => {
        btn.addEventListener('click', function(e) {
            // Let the link handle navigation
        });
    });
    
    // Set initial level from URL or default
    if (currentLevel > 0) {
        currentLevel = currentLevel.toString();
    } else {
        currentLevel = 'all';
    }
    
    // Search functionality
    const searchInput = document.getElementById('search-input');
    const searchBtn = document.getElementById('search-btn');
    
    searchBtn.addEventListener('click', function() {
        currentSearch = searchInput.value.trim();
        currentPage = 1;
        loadVocabulary();
    });
    
    searchInput.addEventListener('keypress', function(e) {
        if (e.key === 'Enter') {
            currentSearch = searchInput.value.trim();
            currentPage = 1;
            loadVocabulary();
        }
    });
    
    // Load initial vocabulary
    loadVocabulary();
});

async function loadVocabulary() {
    const vocabularyList = document.getElementById('vocabulary-list');
    const resultsCount = document.getElementById('results-count');
    const pagination = document.getElementById('pagination');
    
    // Show loading
    vocabularyList.innerHTML = '<div class="loading"><div class="spinner"></div><p>Đang tải từ vựng...</p></div>';
    
    try {
        const params = new URLSearchParams({
            action: 'get_all',
            level: currentLevel === 'all' ? 'all' : currentLevel,
            search: currentSearch,
            page: currentPage,
            limit: itemsPerPage
        });
        
        const data = await apiRequest(`api.php?${params}`);
        if (!data) return; // Redirected to login
        
        if (data.error) {
            vocabularyList.innerHTML = `<p class="loading">Lỗi: ${data.error}</p>`;
            if (resultsCount) {
                resultsCount.textContent = 'Lỗi';
            }
            return;
        }
        
        // Update results count
        if (resultsCount) {
            resultsCount.textContent = `${data.total || 0} từ vựng`;
        }
        
        // Display vocabulary
        vocabularyList.innerHTML = '';
        if (data.data && data.data.length > 0) {
            data.data.forEach(word => {
                const wordCard = createVocabularyCard(word);
                vocabularyList.appendChild(wordCard);
            });
        } else {
            vocabularyList.innerHTML = '<p class="loading">Không tìm thấy từ vựng nào</p>';
        }
        
        // Update pagination
        updatePagination(data.totalPages || 1, pagination);
        
    } catch (error) {
        console.error('Error loading vocabulary:', error);
        vocabularyList.innerHTML = '<p class="loading">Lỗi khi tải dữ liệu. Vui lòng thử lại sau.</p>';
        if (resultsCount) {
            resultsCount.textContent = 'Lỗi';
        }
    }
}

function createVocabularyCard(word) {
    const card = document.createElement('div');
    card.className = 'vocab-card';
    
    const form = word.forms && word.forms[0] ? word.forms[0] : {};
    const pinyin = form.transcriptions ? form.transcriptions.pinyin : '';
    // Prioritize Vietnamese meanings if available
    const meanings = form.meanings_vi && form.meanings_vi.length > 0 
        ? form.meanings_vi.join(', ') 
        : (form.meanings ? form.meanings.join(', ') : '');
    const pos = word.pos ? word.pos.join(', ') : '';
    const level = word.level && word.level.length > 0 ? word.level[0] : '';
    const levelNum = level.replace('new-', '').replace('old-', '').replace('+', '');
    const traditional = form.traditional || '';
    const classifiers = form.classifiers && form.classifiers.length > 0 ? form.classifiers.join(', ') : '';
    
    card.innerHTML = `
        <div class="vocab-header">
            <div class="chinese-char">${word.simplified || ''}</div>
            ${levelNum && levelNum !== '7+' ? `<span class="vocab-level">HSK ${levelNum}</span>` : ''}
        </div>
        ${traditional && traditional !== word.simplified ? `<div style="color: var(--text-secondary); margin-bottom: 0.5rem;">繁體: ${traditional}</div>` : ''}
        <div class="pinyin">${pinyin || ''}</div>
        <div class="meanings">${meanings || ''}</div>
        ${pos ? `<div style="margin-top: 0.5rem;"><span class="pos">${pos}</span></div>` : ''}
        ${classifiers ? `<div style="margin-top: 0.5rem; color: var(--text-secondary); font-size: 0.9rem;">Lượng từ: ${classifiers}</div>` : ''}
        ${word.radical ? `<div style="margin-top: 0.5rem; color: var(--text-secondary); font-size: 0.9rem;">Bộ thủ: ${word.radical}</div>` : ''}
    `;
    
    return card;
}

function updatePagination(totalPages, paginationElement) {
    if (!paginationElement) return;
    
    paginationElement.innerHTML = '';
    
    if (totalPages <= 1) return;
    
    // Previous button
    const prevBtn = document.createElement('button');
    prevBtn.textContent = '← Trước';
    prevBtn.disabled = currentPage === 1;
    prevBtn.addEventListener('click', () => {
        if (currentPage > 1) {
            currentPage--;
            loadVocabulary();
            window.scrollTo({ top: 0, behavior: 'smooth' });
        }
    });
    paginationElement.appendChild(prevBtn);
    
    // Page numbers
    const maxVisiblePages = 5;
    let startPage = Math.max(1, currentPage - Math.floor(maxVisiblePages / 2));
    let endPage = Math.min(totalPages, startPage + maxVisiblePages - 1);
    
    if (endPage - startPage < maxVisiblePages - 1) {
        startPage = Math.max(1, endPage - maxVisiblePages + 1);
    }
    
    if (startPage > 1) {
        const firstBtn = document.createElement('button');
        firstBtn.textContent = '1';
        firstBtn.addEventListener('click', () => {
            currentPage = 1;
            loadVocabulary();
            window.scrollTo({ top: 0, behavior: 'smooth' });
        });
        paginationElement.appendChild(firstBtn);
        
        if (startPage > 2) {
            const ellipsis = document.createElement('span');
            ellipsis.textContent = '...';
            ellipsis.style.padding = '0.5rem';
            paginationElement.appendChild(ellipsis);
        }
    }
    
    for (let i = startPage; i <= endPage; i++) {
        const pageBtn = document.createElement('button');
        pageBtn.textContent = i;
        if (i === currentPage) {
            pageBtn.classList.add('active');
        }
        pageBtn.addEventListener('click', () => {
            currentPage = i;
            loadVocabulary();
            window.scrollTo({ top: 0, behavior: 'smooth' });
        });
        paginationElement.appendChild(pageBtn);
    }
    
    if (endPage < totalPages) {
        if (endPage < totalPages - 1) {
            const ellipsis = document.createElement('span');
            ellipsis.textContent = '...';
            ellipsis.style.padding = '0.5rem';
            paginationElement.appendChild(ellipsis);
        }
        
        const lastBtn = document.createElement('button');
        lastBtn.textContent = totalPages;
        lastBtn.addEventListener('click', () => {
            currentPage = totalPages;
            loadVocabulary();
            window.scrollTo({ top: 0, behavior: 'smooth' });
        });
        paginationElement.appendChild(lastBtn);
    }
    
    // Next button
    const nextBtn = document.createElement('button');
    nextBtn.textContent = 'Sau →';
    nextBtn.disabled = currentPage === totalPages;
    nextBtn.addEventListener('click', () => {
        if (currentPage < totalPages) {
            currentPage++;
            loadVocabulary();
            window.scrollTo({ top: 0, behavior: 'smooth' });
        }
    });
    paginationElement.appendChild(nextBtn);
}

