// Quiz page JavaScript
let quizData = [];
let currentQuestionIndex = 0;
let score = 0;
let correctAnswers = 0;
let wrongAnswers = 0;
let selectedAnswer = null;
let quizType = 'mixed';

document.addEventListener('DOMContentLoaded', function() {
    const startBtn = document.getElementById('start-quiz');
    const nextBtn = document.getElementById('next-question');
    const restartBtn = document.getElementById('restart-quiz');
    
    startBtn.addEventListener('click', startQuiz);
    nextBtn.addEventListener('click', nextQuestion);
    restartBtn.addEventListener('click', function() {
        document.getElementById('quiz-setup').style.display = 'block';
        document.getElementById('quiz-container').style.display = 'none';
        document.getElementById('quiz-result').style.display = 'none';
    });
});

async function startQuiz() {
    const level = document.getElementById('quiz-level').value;
    const count = parseInt(document.getElementById('quiz-count').value);
    quizType = document.getElementById('quiz-type').value;
    
    try {
        const params = new URLSearchParams({
            action: 'get_random',
            level: level,
            count: count
        });
        
        const response = await fetch(`api.php?${params}`);
        if (!response.ok) {
            alert('Lỗi kết nối. Vui lòng thử lại.');
            return;
        }
        const data = await response.json();
        
        if (data.error) {
            alert('Lỗi: ' + data.error);
            return;
        }
        
        if (!data.data || data.data.length === 0) {
            alert('Không tìm thấy từ vựng nào. Vui lòng thử lại với cấp độ khác.');
            return;
        }
        
        if (data.data.length < 4) {
            alert('Cần ít nhất 4 từ vựng để tạo bài kiểm tra. Vui lòng tăng số lượng câu hỏi hoặc chọn cấp độ khác.');
            return;
        }
        
        quizData = data.data;
        currentQuestionIndex = 0;
        score = 0;
        correctAnswers = 0;
        wrongAnswers = 0;
        
        document.getElementById('quiz-setup').style.display = 'none';
        document.getElementById('quiz-container').style.display = 'block';
        document.getElementById('quiz-result').style.display = 'none';
        
        displayQuestion();
    } catch (error) {
        console.error('Error loading quiz:', error);
        alert('Lỗi khi tải dữ liệu. Vui lòng thử lại.');
    }
}

function displayQuestion() {
    if (currentQuestionIndex >= quizData.length) {
        showResults();
        return;
    }
    
    const question = quizData[currentQuestionIndex];
    const form = question.forms && question.forms[0] ? question.forms[0] : {};
    const correctMeaning = form.meanings ? form.meanings[0] : '';
    const correctPinyin = form.transcriptions ? form.transcriptions.pinyin : '';
    
    // Get wrong answers from other words
    const wrongMeanings = [];
    const wrongPinyins = [];
    
    for (let i = 0; i < quizData.length; i++) {
        if (i !== currentQuestionIndex) {
            const otherForm = quizData[i].forms && quizData[i].forms[0] ? quizData[i].forms[0] : {};
            if (otherForm.meanings && otherForm.meanings[0]) {
                wrongMeanings.push(otherForm.meanings[0]);
            }
            if (otherForm.transcriptions && otherForm.transcriptions.pinyin) {
                wrongPinyins.push(otherForm.transcriptions.pinyin);
            }
        }
    }
    
    // Shuffle and select 3 wrong answers
    shuffleArray(wrongMeanings);
    shuffleArray(wrongPinyins);
    
    let questionType = quizType;
    if (quizType === 'mixed') {
        questionType = Math.random() < 0.5 ? 'meaning' : 'pinyin';
    }
    
    // Display question
    const questionDiv = document.getElementById('quiz-question');
    if (questionType === 'meaning') {
        questionDiv.innerHTML = `
            <div class="question-text">${question.simplified}</div>
            <div class="question-pinyin">${correctPinyin}</div>
            <p style="margin-top: 1rem; font-size: 1.1rem; color: var(--text-secondary);">Chọn nghĩa đúng:</p>
        `;
    } else {
        questionDiv.innerHTML = `
            <div class="question-text">${question.simplified}</div>
            <div class="question-pinyin" style="margin-bottom: 1rem;">${correctMeaning}</div>
            <p style="font-size: 1.1rem; color: var(--text-secondary);">Chọn pinyin đúng:</p>
        `;
    }
    
    // Create options
    let options = [];
    if (questionType === 'meaning') {
        options = [correctMeaning, ...wrongMeanings.slice(0, 3)];
    } else {
        options = [correctPinyin, ...wrongPinyins.slice(0, 3)];
    }
    
    shuffleArray(options);
    
    const optionsDiv = document.getElementById('quiz-options');
    optionsDiv.innerHTML = '';
    
    options.forEach((option, index) => {
        const optionBtn = document.createElement('button');
        optionBtn.className = 'option-btn';
        optionBtn.textContent = option;
        optionBtn.addEventListener('click', function() {
            if (selectedAnswer !== null) return; // Already answered
            
            selectedAnswer = option;
            const isCorrect = option === (questionType === 'meaning' ? correctMeaning : correctPinyin);
            
            // Disable all buttons
            document.querySelectorAll('.option-btn').forEach(btn => {
                btn.disabled = true;
                if (btn.textContent === (questionType === 'meaning' ? correctMeaning : correctPinyin)) {
                    btn.classList.add('correct');
                }
                if (btn.textContent === option && !isCorrect) {
                    btn.classList.add('incorrect');
                }
            });
            
            // Show feedback
            const feedbackDiv = document.getElementById('quiz-feedback');
            feedbackDiv.style.display = 'block';
            if (isCorrect) {
                feedbackDiv.className = 'quiz-feedback correct';
                feedbackDiv.textContent = 'Đúng! ✓';
                score++;
                correctAnswers++;
            } else {
                feedbackDiv.className = 'quiz-feedback incorrect';
                feedbackDiv.textContent = `Sai! Đáp án đúng: ${questionType === 'meaning' ? correctMeaning : correctPinyin}`;
                wrongAnswers++;
            }
            
            // Show next button
            document.getElementById('next-question').style.display = 'block';
            
            // Update score
            document.getElementById('quiz-score').textContent = score;
        });
        
        optionsDiv.appendChild(optionBtn);
    });
    
    // Hide feedback and next button
    document.getElementById('quiz-feedback').style.display = 'none';
    document.getElementById('next-question').style.display = 'none';
    selectedAnswer = null;
    
    // Update progress
    const progress = ((currentQuestionIndex + 1) / quizData.length) * 100;
    document.getElementById('quiz-progress-fill').style.width = progress + '%';
    document.getElementById('quiz-current').textContent = currentQuestionIndex + 1;
    document.getElementById('quiz-total').textContent = quizData.length;
}

function nextQuestion() {
    currentQuestionIndex++;
    displayQuestion();
}

async function showResults() {
    document.getElementById('quiz-container').style.display = 'none';
    document.getElementById('quiz-result').style.display = 'block';
    
    const percentage = Math.round((correctAnswers / quizData.length) * 100);
    
    document.getElementById('result-percentage').textContent = percentage;
    document.getElementById('result-correct').textContent = correctAnswers;
    document.getElementById('result-wrong').textContent = wrongAnswers;
    document.getElementById('result-total').textContent = quizData.length;
    
    // Save quiz result if user is logged in
    try {
        const level = document.getElementById('quiz-level').value;
        const response = await fetch('api.php?action=save_quiz_result', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                hsk_level: level,
                total_questions: quizData.length,
                correct_answers: correctAnswers,
                score: percentage,
                quiz_type: quizType
            })
        });
        
        const result = await response.json();
        if (result.error && result.error !== 'Not logged in') {
            console.error('Error saving quiz result:', result.error);
        }
    } catch (error) {
        console.error('Error saving quiz result:', error);
    }
}

function shuffleArray(array) {
    for (let i = array.length - 1; i > 0; i--) {
        const j = Math.floor(Math.random() * (i + 1));
        [array[i], array[j]] = [array[j], array[i]];
    }
}

