// Main JavaScript for homepage
document.addEventListener('DOMContentLoaded', function() {
    // Initialize mobile menu
    initMobileMenu();
    
    // Only load stats and random words if user is logged in
    // Check if we're on the homepage (has total-words element)
    const totalWordsEl = document.getElementById('total-words');
    if (totalWordsEl) {
        loadStats();
    }
    
    // Load random words if element exists
    const randomWordsEl = document.getElementById('random-words');
    if (randomWordsEl) {
        loadRandomWords();
    }
    
    // Load HSK level stats if elements exist (for homepage hero section)
    const hsk1WordsEl = document.getElementById('hsk1-words');
    if (hsk1WordsEl) {
        loadHSKLevelStats();
    }
});

// Mobile Menu Functionality
function initMobileMenu() {
    const menuToggle = document.querySelector('.mobile-menu-toggle');
    const navMenu = document.getElementById('navMenu');
    const userMenu = document.querySelector('.user-menu');
    
    if (!menuToggle || !navMenu) return;
    
    // Toggle menu on button click
    menuToggle.addEventListener('click', function(e) {
        e.stopPropagation();
        navMenu.classList.toggle('active');
        menuToggle.classList.toggle('active');
        
        // Update menu icon
        const icon = menuToggle.querySelector('.menu-icon');
        if (icon) {
            icon.textContent = navMenu.classList.contains('active') ? '✕' : '☰';
        }
    });
    
    // Close menu when clicking outside
    document.addEventListener('click', function(e) {
        if (!navMenu.contains(e.target) && !menuToggle.contains(e.target)) {
            navMenu.classList.remove('active');
            menuToggle.classList.remove('active');
            const icon = menuToggle.querySelector('.menu-icon');
            if (icon) {
                icon.textContent = '☰';
            }
        }
    });
    
    // Close menu when clicking on a link
    const navLinks = navMenu.querySelectorAll('a');
    navLinks.forEach(link => {
        link.addEventListener('click', function() {
            navMenu.classList.remove('active');
            menuToggle.classList.remove('active');
            const icon = menuToggle.querySelector('.menu-icon');
            if (icon) {
                icon.textContent = '☰';
            }
        });
    });
    
    // Handle user menu on mobile
    if (userMenu) {
        const userMenuToggle = userMenu.querySelector('.user-menu-toggle');
        if (userMenuToggle) {
            userMenuToggle.addEventListener('click', function(e) {
                if (window.innerWidth <= 768) {
                    e.preventDefault();
                    userMenu.classList.toggle('active');
                }
            });
        }
    }
    
    // Close menu on window resize
    window.addEventListener('resize', function() {
        if (window.innerWidth > 768) {
            navMenu.classList.remove('active');
            menuToggle.classList.remove('active');
            const icon = menuToggle.querySelector('.menu-icon');
            if (icon) {
                icon.textContent = '☰';
            }
        }
    });
}

async function loadStats() {
    try {
        const data = await apiRequest('api.php?action=get_stats');
        if (!data) return; // Redirected to login
        
        if (data.total) {
            document.getElementById('total-words').textContent = formatNumber(data.total);
        }
        
        if (data.by_level) {
            const hsk1El = document.getElementById('hsk1-words');
            const hsk2El = document.getElementById('hsk2-words');
            const hsk3El = document.getElementById('hsk3-words');
            const hsk4El = document.getElementById('hsk4-words');
            
            if (hsk1El) hsk1El.textContent = formatNumber(data.by_level[1] || 0) + ' từ vựng';
            if (hsk2El) hsk2El.textContent = formatNumber(data.by_level[2] || 0) + ' từ vựng';
            if (hsk3El) hsk3El.textContent = formatNumber(data.by_level[3] || 0) + ' từ vựng';
            if (hsk4El) hsk4El.textContent = formatNumber(data.by_level[4] || 0) + ' từ vựng';
        }
    } catch (error) {
        console.error('Error loading stats:', error);
        // Set default values on error
        document.getElementById('total-words').textContent = '0';
        for (let i = 1; i <= 4; i++) {
            const elem = document.getElementById(`hsk${i}-words`);
            if (elem) elem.textContent = '0';
        }
    }
}

async function loadRandomWords() {
    try {
        const data = await apiRequest('api.php?action=get_random&count=6');
        if (!data) return; // Redirected to login
        
        const wordsGrid = document.getElementById('random-words');
        if (!wordsGrid) return;
        
        wordsGrid.innerHTML = '';
        
        if (data.data && data.data.length > 0) {
            data.data.forEach(word => {
                const wordCard = createWordCard(word);
                wordsGrid.appendChild(wordCard);
            });
        } else {
            wordsGrid.innerHTML = '<p class="loading">Không có dữ liệu</p>';
        }
    } catch (error) {
        console.error('Error loading random words:', error);
        const wordsGrid = document.getElementById('random-words');
        if (wordsGrid) {
            wordsGrid.innerHTML = '<p class="loading">Lỗi khi tải dữ liệu</p>';
        }
    }
}

function createWordCard(word) {
    const card = document.createElement('div');
    card.className = 'vocab-card';
    
    const form = word.forms && word.forms[0] ? word.forms[0] : {};
    const pinyin = form.transcriptions ? form.transcriptions.pinyin : '';
    // Prioritize Vietnamese meanings if available
    const meanings = form.meanings_vi && form.meanings_vi.length > 0
        ? form.meanings_vi.slice(0, 2).join(', ')
        : (form.meanings ? form.meanings.slice(0, 2).join(', ') : '');
    const pos = word.pos ? word.pos.join(', ') : '';
    const level = word.level && word.level.length > 0 ? word.level[0] : '';
    const levelNum = level.replace('new-', '').replace('old-', '').replace('+', '');
    
    card.innerHTML = `
        <div class="vocab-header">
            <div class="chinese-char">${word.simplified || ''}</div>
            ${levelNum && levelNum !== '7+' ? `<span class="vocab-level">HSK ${levelNum}</span>` : ''}
        </div>
        <div class="pinyin">${pinyin || ''}</div>
        <div class="meanings">${meanings || ''}</div>
        ${pos ? `<div class="pos">${pos}</div>` : ''}
    `;
    
    return card;
}

async function loadHSKLevelStats() {
    try {
        const data = await apiRequest('api.php?action=get_stats');
        if (!data) {
            // User not logged in, show placeholder
            document.getElementById('hsk1-words').textContent = 'Đăng nhập để xem';
            document.getElementById('hsk2-words').textContent = 'Đăng nhập để xem';
            document.getElementById('hsk3-words').textContent = 'Đăng nhập để xem';
            document.getElementById('hsk4-words').textContent = 'Đăng nhập để xem';
            return;
        }
        
        if (data.by_level) {
            document.getElementById('hsk1-words').textContent = formatNumber(data.by_level[1] || 0) + ' từ vựng';
            document.getElementById('hsk2-words').textContent = formatNumber(data.by_level[2] || 0) + ' từ vựng';
            document.getElementById('hsk3-words').textContent = formatNumber(data.by_level[3] || 0) + ' từ vựng';
            document.getElementById('hsk4-words').textContent = formatNumber(data.by_level[4] || 0) + ' từ vựng';
        }
    } catch (error) {
        console.error('Error loading HSK level stats:', error);
        // Show placeholder on error
        const hskElements = ['hsk1-words', 'hsk2-words', 'hsk3-words', 'hsk4-words'];
        hskElements.forEach(id => {
            const el = document.getElementById(id);
            if (el) el.textContent = '-';
        });
    }
}

function formatNumber(num) {
    return num.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ",");
}

