// Flashcard page JavaScript
let flashcards = [];
let currentIndex = 0;
let isFlipped = false;
let knownCards = [];
let unknownCards = [];

document.addEventListener('DOMContentLoaded', function() {
    const startBtn = document.getElementById('start-flashcard');
    const flipBtn = document.getElementById('flip-card');
    const prevBtn = document.getElementById('prev-card');
    const nextBtn = document.getElementById('next-card');
    const knowBtn = document.getElementById('know-card');
    const dontKnowBtn = document.getElementById('dont-know-card');
    const flashcard = document.getElementById('flashcard');
    
    startBtn.addEventListener('click', startFlashcard);
    flipBtn.addEventListener('click', flipCard);
    prevBtn.addEventListener('click', previousCard);
    nextBtn.addEventListener('click', nextCard);
    knowBtn.addEventListener('click', markAsKnown);
    dontKnowBtn.addEventListener('click', markAsUnknown);
    
    // Keyboard shortcuts
    document.addEventListener('keydown', function(e) {
        if (document.getElementById('flashcard-container').style.display === 'none') return;
        
        switch(e.key) {
            case 'ArrowLeft':
                e.preventDefault();
                previousCard();
                break;
            case 'ArrowRight':
                e.preventDefault();
                nextCard();
                break;
            case ' ':
            case 'Enter':
                e.preventDefault();
                flipCard();
                break;
            case '1':
                e.preventDefault();
                markAsKnown();
                break;
            case '2':
                e.preventDefault();
                markAsUnknown();
                break;
        }
    });
});

async function startFlashcard() {
    const level = window.flashcardLevel || 'all';
    const count = parseInt(document.getElementById('flashcard-count').value);
    
    try {
        const params = new URLSearchParams({
            action: 'get_random',
            level: level.toString(),
            count: count
        });
        
        const data = await apiRequest(`api.php?${params}`);
        if (!data) return; // Redirected to login
        
        if (data.error) {
            alert('Lỗi: ' + data.error);
            return;
        }
        
        if (!data.data || data.data.length === 0) {
            alert('Không tìm thấy từ vựng nào. Vui lòng thử lại với cấp độ khác.');
            return;
        }
        
        flashcards = data.data;
        currentIndex = 0;
        isFlipped = false;
        knownCards = [];
        unknownCards = [];
        
        document.getElementById('flashcard-container').style.display = 'block';
        updateCard();
        updateStats();
    } catch (error) {
        console.error('Error loading flashcards:', error);
        alert('Lỗi khi tải dữ liệu. Vui lòng thử lại.');
    }
}

function updateCard() {
    if (flashcards.length === 0) return;
    
    const word = flashcards[currentIndex];
    const form = word.forms && word.forms[0] ? word.forms[0] : {};
    const pinyin = form.transcriptions ? form.transcriptions.pinyin : '';
    // Prioritize Vietnamese meanings if available
    const meanings = form.meanings_vi && form.meanings_vi.length > 0
        ? form.meanings_vi.join(', ')
        : (form.meanings ? form.meanings.join(', ') : '');
    const pos = word.pos ? word.pos.join(', ') : '';
    
    // Front side
    document.getElementById('chinese-char').textContent = word.simplified || '';
    document.getElementById('pinyin-text').textContent = pinyin || '';
    
    // Back side
    document.getElementById('chinese-char-back').textContent = word.simplified || '';
    document.getElementById('pinyin-text-back').textContent = pinyin || '';
    document.getElementById('meaning-text').textContent = meanings || '';
    document.getElementById('pos-text').textContent = pos ? `(${pos})` : '';
    
    // Reset flip state
    isFlipped = false;
    document.getElementById('flashcard').classList.remove('flipped');
    
    // Update progress
    const progress = ((currentIndex + 1) / flashcards.length) * 100;
    document.getElementById('progress-fill').style.width = progress + '%';
    document.getElementById('current-card').textContent = currentIndex + 1;
    document.getElementById('total-cards').textContent = flashcards.length;
}

function flipCard() {
    isFlipped = !isFlipped;
    const flashcard = document.getElementById('flashcard');
    if (isFlipped) {
        flashcard.classList.add('flipped');
    } else {
        flashcard.classList.remove('flipped');
    }
}

function previousCard() {
    if (currentIndex > 0) {
        currentIndex--;
        updateCard();
    }
}

function nextCard() {
    if (currentIndex < flashcards.length - 1) {
        currentIndex++;
        updateCard();
    }
}

function markAsKnown() {
    if (flashcards.length === 0) return;
    
    const wordId = flashcards[currentIndex].simplified;
    if (!knownCards.includes(wordId)) {
        knownCards.push(wordId);
        if (unknownCards.includes(wordId)) {
            unknownCards = unknownCards.filter(id => id !== wordId);
        }
    }
    
    updateStats();
    
    // Auto move to next card
    setTimeout(() => {
        if (currentIndex < flashcards.length - 1) {
            nextCard();
        } else {
            // Show completion message
            alert(`Hoàn thành! Đã biết: ${knownCards.length}, Chưa biết: ${unknownCards.length}`);
        }
    }, 300);
}

function markAsUnknown() {
    if (flashcards.length === 0) return;
    
    const wordId = flashcards[currentIndex].simplified;
    if (!unknownCards.includes(wordId)) {
        unknownCards.push(wordId);
        if (knownCards.includes(wordId)) {
            knownCards = knownCards.filter(id => id !== wordId);
        }
    }
    
    updateStats();
    
    // Auto move to next card
    setTimeout(() => {
        if (currentIndex < flashcards.length - 1) {
            nextCard();
        } else {
            // Show completion message
            alert(`Hoàn thành! Đã biết: ${knownCards.length}, Chưa biết: ${unknownCards.length}`);
        }
    }, 300);
}

function updateStats() {
    document.getElementById('known-count').textContent = knownCards.length;
    document.getElementById('unknown-count').textContent = unknownCards.length;
}

