// Exercises page JavaScript
let exerciseData = [];
let currentExerciseIndex = 0;
let currentExerciseType = 'fill-blank';
let correctCount = 0;
let wrongCount = 0;
let totalCount = 0;
let currentExercise = null;

document.addEventListener('DOMContentLoaded', function() {
    const level = window.exerciseLevel || 0;
    
    if (level === 0) {
        return; // Level selector page
    }
    
    // Exercise type buttons
    document.querySelectorAll('.exercise-type-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            document.querySelectorAll('.exercise-type-btn').forEach(b => b.classList.remove('active'));
            this.classList.add('active');
            currentExerciseType = this.dataset.type;
            loadExercise();
        });
    });
    
    // Action buttons
    document.getElementById('check-answer')?.addEventListener('click', checkAnswer);
    document.getElementById('next-exercise')?.addEventListener('click', nextExercise);
    document.getElementById('reset-exercise')?.addEventListener('click', resetExercise);
    
    // Load initial exercise
    loadExercise();
});

async function loadExercise() {
    const level = window.exerciseLevel || 1;
    const exerciseContent = document.getElementById('exercise-content');
    
    try {
        // Get random vocabulary for exercises
        const data = await apiRequest(`api.php?action=get_random&level=${level}&count=20`);
        if (!data) return; // Redirected to login
        
        if (data.error || !data.data || data.data.length === 0) {
            exerciseContent.innerHTML = '<p class="loading">Không tìm thấy từ vựng. Vui lòng thử lại.</p>';
            return;
        }
        
        exerciseData = data.data;
        currentExerciseIndex = 0;
        correctCount = 0;
        wrongCount = 0;
        totalCount = 0;
        
        updateStats();
        generateExercise();
        
    } catch (error) {
        console.error('Error loading exercise:', error);
        exerciseContent.innerHTML = '<p class="loading">Lỗi khi tải bài tập. Vui lòng thử lại.</p>';
    }
}

function generateExercise() {
    if (currentExerciseIndex >= exerciseData.length) {
        showCompletion();
        return;
    }
    
    currentExercise = exerciseData[currentExerciseIndex];
    const form = currentExercise.forms && currentExercise.forms[0] ? currentExercise.forms[0] : {};
    const pinyin = form.transcriptions ? form.transcriptions.pinyin : '';
    // Prioritize Vietnamese meanings if available
    const meanings = form.meanings_vi && form.meanings_vi.length > 0
        ? form.meanings_vi.join(', ')
        : (form.meanings ? form.meanings.join(', ') : '');
    
    const exerciseContent = document.getElementById('exercise-content');
    const checkBtn = document.getElementById('check-answer');
    const nextBtn = document.getElementById('next-exercise');
    
    checkBtn.style.display = 'block';
    nextBtn.style.display = 'none';
    
    switch (currentExerciseType) {
        case 'fill-blank':
            generateFillBlank(exerciseContent, currentExercise, pinyin, meanings);
            break;
        case 'multiple-choice':
            generateMultipleChoice(exerciseContent, currentExercise, pinyin, meanings);
            break;
        case 'translation':
            generateTranslation(exerciseContent, currentExercise, pinyin, meanings);
            break;
    }
    
    totalCount++;
    updateStats();
}

function generateFillBlank(container, word, pinyin, meanings) {
    // Get the first meaning as correct answer
    const form = word.forms && word.forms[0] ? word.forms[0] : {};
    const correctAnswer = form.meanings_vi && form.meanings_vi.length > 0
        ? form.meanings_vi[0]
        : (form.meanings && form.meanings.length > 0 ? form.meanings[0] : meanings);
    
    container.innerHTML = `
        <div class="exercise-question">
            <h3>Điền từ vào chỗ trống</h3>
            <div class="question-text">
                <div class="chinese-char" style="font-size: 3rem; margin: 1rem 0;">${word.simplified}</div>
                <div class="pinyin" style="font-size: 1.5rem; color: var(--primary-color); margin-bottom: 1rem;">${pinyin}</div>
                <p style="font-size: 1.2rem; margin-bottom: 1.5rem;">Nghĩa của từ này là gì? (Nhập bằng tiếng Việt)</p>
                <input type="text" id="exercise-answer" class="exercise-input" placeholder="Nhập nghĩa tiếng Việt của từ...">
            </div>
        </div>
    `;
    
    currentExercise.correctAnswer = correctAnswer;
}

function generateMultipleChoice(container, word, pinyin, meanings) {
    // Get correct answer (prefer Vietnamese)
    const form = word.forms && word.forms[0] ? word.forms[0] : {};
    const correctAnswer = form.meanings_vi && form.meanings_vi.length > 0
        ? form.meanings_vi[0]
        : (form.meanings && form.meanings.length > 0 ? form.meanings[0] : meanings.split(',')[0]);
    
    // Get wrong answers (prefer Vietnamese)
    const wrongAnswers = [];
    for (let i = 0; i < exerciseData.length; i++) {
        if (i !== currentExerciseIndex) {
            const otherForm = exerciseData[i].forms && exerciseData[i].forms[0] ? exerciseData[i].forms[0] : {};
            let wrongAnswer = '';
            if (otherForm.meanings_vi && otherForm.meanings_vi.length > 0) {
                wrongAnswer = otherForm.meanings_vi[0];
            } else if (otherForm.meanings && otherForm.meanings.length > 0) {
                wrongAnswer = otherForm.meanings[0];
            }
            if (wrongAnswer) {
                wrongAnswers.push(wrongAnswer);
            }
        }
    }
    
    // Shuffle and select 3 wrong answers
    shuffleArray(wrongAnswers);
    const options = [correctAnswer, ...wrongAnswers.slice(0, 3)];
    shuffleArray(options);
    
    container.innerHTML = `
        <div class="exercise-question">
            <h3>Chọn đáp án đúng</h3>
            <div class="question-text">
                <div class="chinese-char" style="font-size: 3rem; margin: 1rem 0;">${word.simplified}</div>
                <div class="pinyin" style="font-size: 1.5rem; color: var(--primary-color); margin-bottom: 1.5rem;">${pinyin}</div>
                <p style="font-size: 1.1rem; margin-bottom: 1rem;">Chọn nghĩa đúng (tiếng Việt):</p>
                <div class="exercise-options">
                    ${options.map((opt, index) => `
                        <button class="option-btn" data-answer="${opt}">${opt}</button>
                    `).join('')}
                </div>
            </div>
        </div>
    `;
    
    // Add click handlers
    document.querySelectorAll('.option-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            document.querySelectorAll('.option-btn').forEach(b => {
                b.classList.remove('selected');
                b.disabled = true;
            });
            this.classList.add('selected');
            currentExercise.userAnswer = this.dataset.answer;
        });
    });
    
    currentExercise.correctAnswer = correctAnswer;
}

function generateTranslation(container, word, pinyin, meanings) {
    // Get correct answer (prefer Vietnamese, but for translation exercise, use first meaning)
    const form = word.forms && word.forms[0] ? word.forms[0] : {};
    const correctAnswer = form.meanings_vi && form.meanings_vi.length > 0
        ? form.meanings_vi.join(', ')
        : (form.meanings && form.meanings.length > 0 ? form.meanings[0] : meanings);
    
    container.innerHTML = `
        <div class="exercise-question">
            <h3>Dịch câu</h3>
            <div class="question-text">
                <p style="font-size: 1.2rem; margin-bottom: 1.5rem;">Viết nghĩa tiếng Việt của từ:</p>
                <div class="chinese-char" style="font-size: 3rem; margin: 1rem 0;">${word.simplified}</div>
                <div class="pinyin" style="font-size: 1.5rem; color: var(--primary-color); margin-bottom: 1.5rem;">${pinyin}</div>
                <textarea id="exercise-answer" class="exercise-textarea" placeholder="Nhập nghĩa tiếng Việt..." rows="3"></textarea>
            </div>
        </div>
    `;
    
    currentExercise.correctAnswer = correctAnswer;
}

function checkAnswer() {
    if (!currentExercise) return;
    
    const checkBtn = document.getElementById('check-answer');
    const nextBtn = document.getElementById('next-exercise');
    let userAnswer = '';
    let isCorrect = false;
    
    if (currentExerciseType === 'multiple-choice') {
        const selected = document.querySelector('.option-btn.selected');
        if (!selected) {
            alert('Vui lòng chọn đáp án!');
            return;
        }
        userAnswer = selected.dataset.answer;
        // Compare with correct answer (case insensitive, trim)
        const correctAns = currentExercise.correctAnswer.trim().toLowerCase();
        isCorrect = userAnswer.trim().toLowerCase() === correctAns;
        
        // Show correct/incorrect
        document.querySelectorAll('.option-btn').forEach(btn => {
            if (btn.dataset.answer.trim().toLowerCase() === correctAns) {
                btn.classList.add('correct');
            }
            if (btn.classList.contains('selected') && !isCorrect) {
                btn.classList.add('incorrect');
            }
        });
    } else {
        const answerInput = document.getElementById('exercise-answer');
        if (!answerInput || !answerInput.value.trim()) {
            alert('Vui lòng nhập đáp án!');
            return;
        }
        userAnswer = answerInput.value.trim();
        // For fill-blank and translation, compare with first meaning
        const correctAns = currentExercise.correctAnswer.split(',')[0].trim().toLowerCase();
        isCorrect = userAnswer.toLowerCase() === correctAns || 
                   userAnswer.toLowerCase().includes(correctAns) ||
                   correctAns.includes(userAnswer.toLowerCase());
        
        // Show feedback
        const feedback = document.createElement('div');
        feedback.className = isCorrect ? 'exercise-feedback correct' : 'exercise-feedback incorrect';
        feedback.innerHTML = isCorrect 
            ? '✓ Đúng!' 
            : `✗ Sai! Đáp án đúng: ${currentExercise.correctAnswer.split(',')[0]}`;
        answerInput.parentElement.appendChild(feedback);
        answerInput.disabled = true;
    }
    
    if (isCorrect) {
        correctCount++;
    } else {
        wrongCount++;
    }
    
    updateStats();
    checkBtn.style.display = 'none';
    nextBtn.style.display = 'block';
    
    // Save exercise result if logged in
    saveExerciseResult(userAnswer, currentExercise.correctAnswer, isCorrect);
}

function nextExercise() {
    currentExerciseIndex++;
    generateExercise();
}

function resetExercise() {
    loadExercise();
}

function showCompletion() {
    const exerciseContent = document.getElementById('exercise-content');
    exerciseContent.innerHTML = `
        <div class="exercise-completion">
            <h2>Hoàn thành bài tập!</h2>
            <div class="completion-stats">
                <p>Đúng: <strong>${correctCount}</strong></p>
                <p>Sai: <strong>${wrongCount}</strong></p>
                <p>Tổng: <strong>${totalCount}</strong> câu</p>
                <p>Điểm: <strong>${totalCount > 0 ? Math.round((correctCount / totalCount) * 100) : 0}%</strong></p>
            </div>
            <button onclick="resetExercise()" class="btn btn-primary">Làm Lại</button>
        </div>
    `;
    document.getElementById('check-answer').style.display = 'none';
    document.getElementById('next-exercise').style.display = 'none';
}

function updateStats() {
    document.getElementById('correct-count').textContent = correctCount;
    document.getElementById('wrong-count').textContent = wrongCount;
    document.getElementById('total-count').textContent = totalCount;
}

function shuffleArray(array) {
    for (let i = array.length - 1; i > 0; i--) {
        const j = Math.floor(Math.random() * (i + 1));
        [array[i], array[j]] = [array[j], array[i]];
    }
}

async function saveExerciseResult(userAnswer, correctAnswer, isCorrect) {
    // This would save to database if user is logged in
    // Implementation similar to quiz results
}

// Add CSS for exercise elements
const style = document.createElement('style');
style.textContent = `
    .exercise-question {
        text-align: center;
    }
    .exercise-input, .exercise-textarea {
        width: 100%;
        padding: 1rem;
        border: 2px solid var(--border-color);
        border-radius: 8px;
        font-size: 1.1rem;
        margin-top: 1rem;
    }
    .exercise-input:focus, .exercise-textarea:focus {
        outline: none;
        border-color: var(--primary-color);
    }
    .exercise-options {
        display: grid;
        gap: 1rem;
        margin-top: 1rem;
    }
    .option-btn {
        padding: 1rem;
        border: 2px solid var(--border-color);
        background: white;
        border-radius: 8px;
        cursor: pointer;
        transition: all 0.3s;
        font-size: 1.1rem;
        text-align: left;
    }
    .option-btn:hover:not(:disabled) {
        border-color: var(--primary-color);
        background: #fff5f5;
    }
    .option-btn.selected {
        border-color: var(--primary-color);
        background: #fff5f5;
    }
    .option-btn.correct {
        background: #e8f5e9;
        border-color: var(--success-color);
    }
    .option-btn.incorrect {
        background: #ffebee;
        border-color: var(--danger-color);
    }
    .option-btn:disabled {
        cursor: not-allowed;
    }
    .exercise-feedback {
        margin-top: 1rem;
        padding: 1rem;
        border-radius: 8px;
        font-weight: 500;
    }
    .exercise-feedback.correct {
        background: #e8f5e9;
        color: var(--success-color);
    }
    .exercise-feedback.incorrect {
        background: #ffebee;
        color: var(--danger-color);
    }
    .exercise-completion {
        text-align: center;
        padding: 2rem;
    }
    .completion-stats {
        margin: 2rem 0;
        font-size: 1.2rem;
    }
`;
document.head.appendChild(style);

