<?php
require_once __DIR__ . '/config.php';
startSession();

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST');
header('Access-Control-Allow-Headers: Content-Type');

/**
 * Check if user is logged in for protected actions
 */
function checkAuth() {
    if (!isLoggedIn()) {
        echo json_encode(['error' => 'Authentication required', 'auth_required' => true]);
        exit;
    }
}

/**
 * Load vocabulary from HSK files (hsk1.json, hsk2.json, hsk3.json, hsk4.json)
 */
function loadHSKData() {
    static $cachedData = null;
    
    if ($cachedData !== null) {
        return $cachedData;
    }
    
    $hskDir = __DIR__ . '/hsk';
    $allData = [];
    
    // Load data from each HSK level file
    for ($i = 1; $i <= 4; $i++) {
        $hskFile = $hskDir . "/hsk$i.json";
        if (file_exists($hskFile)) {
            $content = @file_get_contents($hskFile);
            if ($content !== false) {
                $levelData = json_decode($content, true);
                if ($levelData && is_array($levelData)) {
                    // Convert HSK format to standard format
                    foreach ($levelData as $word) {
                        if (empty($word['simplified'])) continue; // Skip empty entries
                        
                        // Clean pinyin (remove content in brackets like [ái tình])
                        $pinyin = $word['pinyin'] ?? '';
                        if (preg_match('/^(.+?)\s*\[.*?\]\s*$/', $pinyin, $matches)) {
                            $pinyin = trim($matches[1]);
                        }
                        
                        $convertedWord = [
                            'simplified' => $word['simplified'],
                            'level' => ["new-$i"],
                            'forms' => [[
                                'traditional' => $word['simplified'], // Use simplified as traditional if not provided
                                'transcriptions' => [
                                    'pinyin' => $pinyin
                                ],
                                'meanings' => [$word['meaning'] ?? ''],
                                'classifiers' => []
                            ]],
                            'pos' => [],
                            'frequency' => 0,
                            'radical' => ''
                        ];
                        
                        // Add example if available
                        if (!empty($word['example'])) {
                            $convertedWord['example'] = $word['example'];
                            $convertedWord['example_pinyin'] = $word['example_pinyin'] ?? '';
                            $convertedWord['example_meaning'] = $word['example_meaning'] ?? '';
                        }
                        
                        $allData[] = $convertedWord;
                    }
                }
            }
        }
    }
    
    $cachedData = $allData;
    return $allData;
}

/**
 * Get all vocabulary data
 */
$data = loadHSKData();

if (empty($data)) {
    echo json_encode(['error' => 'No vocabulary data found. Please check hsk/*.json files']);
    exit;
}

$action = $_GET['action'] ?? '';

switch ($action) {
    case 'get_all':
        checkAuth(); // Require login
        // Get all vocabulary
        $level = $_GET['level'] ?? '';
        $search = $_GET['search'] ?? '';
        $page = intval($_GET['page'] ?? 1);
        $limit = intval($_GET['limit'] ?? 50);
        
        $filtered = $data;
        
        // Filter by level (HSK 1-4)
        if ($level && $level !== 'all') {
            $levelNum = intval($level);
            if ($levelNum >= 1 && $levelNum <= 4) {
                $filtered = array_filter($filtered, function($item) use ($levelNum) {
                    return in_array("new-$levelNum", $item['level'] ?? []);
                });
            }
        }
        
        // Filter by search
        if ($search) {
            $searchLower = mb_strtolower($search, 'UTF-8');
            $filtered = array_filter($filtered, function($item) use ($searchLower) {
                $simplified = mb_strtolower($item['simplified'] ?? '', 'UTF-8');
                $meanings = '';
                if (isset($item['forms'][0]['meanings']) && is_array($item['forms'][0]['meanings'])) {
                    $meanings = mb_strtolower(implode(' ', $item['forms'][0]['meanings']), 'UTF-8');
                }
                $pinyin = '';
                if (isset($item['forms'][0]['transcriptions']['pinyin'])) {
                    $pinyin = mb_strtolower($item['forms'][0]['transcriptions']['pinyin'], 'UTF-8');
                }
                return strpos($simplified, $searchLower) !== false || 
                       strpos($meanings, $searchLower) !== false ||
                       strpos($pinyin, $searchLower) !== false;
            });
        }
        
        $filtered = array_values($filtered);
        $total = count($filtered);
        $offset = ($page - 1) * $limit;
        $paginated = array_slice($filtered, $offset, $limit);
        
        echo json_encode([
            'data' => $paginated,
            'total' => $total,
            'page' => $page,
            'limit' => $limit,
            'totalPages' => ceil($total / $limit)
        ], JSON_UNESCAPED_UNICODE);
        break;
        
    case 'get_by_level':
        checkAuth(); // Require login
        // Get vocabulary by HSK level
        $level = $_GET['level'] ?? '';
        if (!$level || !is_numeric($level)) {
            echo json_encode(['error' => 'Invalid level parameter']);
            exit;
        }
        
        $levelNum = intval($level);
        if ($levelNum < 1 || $levelNum > 4) {
            echo json_encode(['error' => 'Level must be between 1 and 4']);
            exit;
        }
        
        $filtered = array_filter($data, function($item) use ($levelNum) {
            return in_array("new-$levelNum", $item['level'] ?? []);
        });
        
        echo json_encode([
            'data' => array_values($filtered),
            'total' => count($filtered)
        ], JSON_UNESCAPED_UNICODE);
        break;
        
    case 'get_stats':
        checkAuth(); // Require login
        // Get statistics for HSK 1-4
        $stats = [
            'total' => 0,
            'by_level' => []
        ];
        
        for ($i = 1; $i <= 4; $i++) {
            $count = count(array_filter($data, function($item) use ($i) {
                return in_array("new-$i", $item['level'] ?? []);
            }));
            $stats['by_level'][$i] = $count;
            $stats['total'] += $count;
        }
        
        echo json_encode($stats);
        break;
        
    case 'get_random':
        checkAuth(); // Require login
        // Get random vocabulary for quiz/flashcard
        $level = $_GET['level'] ?? '';
        $count = intval($_GET['count'] ?? 10);
        
        $filtered = $data;
        if ($level && $level !== 'all') {
            $levelNum = intval($level);
            if ($levelNum >= 1 && $levelNum <= 4) {
                $filtered = array_filter($filtered, function($item) use ($levelNum) {
                    return in_array("new-$levelNum", $item['level'] ?? []);
                });
            }
        }
        
        $filtered = array_values($filtered);
        shuffle($filtered);
        $random = array_slice($filtered, 0, min($count, count($filtered)));
        
        echo json_encode(['data' => $random], JSON_UNESCAPED_UNICODE);
        break;
        
    case 'save_quiz_result':
        checkAuth(); // Require login
        // Save quiz result
        
        if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
            echo json_encode(['error' => 'Invalid method']);
            exit;
        }
        
        $input = json_decode(file_get_contents('php://input'), true);
        $user_id = $_SESSION['user_id'];
        $hsk_level = intval($input['hsk_level'] ?? 0);
        $total_questions = intval($input['total_questions'] ?? 0);
        $correct_answers = intval($input['correct_answers'] ?? 0);
        $score = intval($input['score'] ?? 0);
        $quiz_type = $input['quiz_type'] ?? 'mixed';
        
        $conn = getDB();
        $stmt = $conn->prepare("INSERT INTO quiz_results (user_id, hsk_level, total_questions, correct_answers, score, quiz_type) VALUES (?, ?, ?, ?, ?, ?)");
        $stmt->bind_param("iiiiis", $user_id, $hsk_level, $total_questions, $correct_answers, $score, $quiz_type);
        
        if ($stmt->execute()) {
            echo json_encode(['success' => true, 'id' => $conn->insert_id]);
        } else {
            echo json_encode(['error' => 'Failed to save result']);
        }
        break;
        
    default:
        echo json_encode(['error' => 'Invalid action']);
        break;
}
?>
