<?php
require_once '../config.php';
requireAdmin();

$conn = getDB();

// Handle actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $user_id = intval($_POST['user_id'] ?? 0);
    
    if ($action === 'ban' && $user_id > 0) {
        $stmt = $conn->prepare("UPDATE users SET status = 'banned' WHERE id = ? AND id != ?");
        $current_user_id = $_SESSION['user_id'];
        $stmt->bind_param("ii", $user_id, $current_user_id);
        $stmt->execute();
        header('Location: users.php?msg=banned');
        exit;
    } elseif ($action === 'unban' && $user_id > 0) {
        $stmt = $conn->prepare("UPDATE users SET status = 'active' WHERE id = ?");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        header('Location: users.php?msg=unbanned');
        exit;
    } elseif ($action === 'delete' && $user_id > 0) {
        $stmt = $conn->prepare("DELETE FROM users WHERE id = ? AND id != ?");
        $current_user_id = $_SESSION['user_id'];
        $stmt->bind_param("ii", $user_id, $current_user_id);
        $stmt->execute();
        header('Location: users.php?msg=deleted');
        exit;
    } elseif ($action === 'make_admin' && $user_id > 0) {
        $stmt = $conn->prepare("UPDATE users SET is_admin = 1 WHERE id = ?");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        header('Location: users.php?msg=admin_added');
        exit;
    } elseif ($action === 'remove_admin' && $user_id > 0) {
        $stmt = $conn->prepare("UPDATE users SET is_admin = 0 WHERE id = ? AND id != ?");
        $current_user_id = $_SESSION['user_id'];
        $stmt->bind_param("ii", $user_id, $current_user_id);
        $stmt->execute();
        header('Location: users.php?msg=admin_removed');
        exit;
    }
}

// Get search and filter
$search = $_GET['search'] ?? '';
$status_filter = $_GET['status'] ?? 'all';
$role_filter = $_GET['role'] ?? 'all';
$page = intval($_GET['page'] ?? 1);
$limit = 20;
$offset = ($page - 1) * $limit;

// Build query
$where = [];
$params = [];
$types = '';

if ($search) {
    $where[] = "(username LIKE ? OR email LIKE ? OR full_name LIKE ?)";
    $searchParam = "%$search%";
    $params[] = $searchParam;
    $params[] = $searchParam;
    $params[] = $searchParam;
    $types .= 'sss';
}

if ($status_filter !== 'all') {
    $where[] = "status = ?";
    $params[] = $status_filter;
    $types .= 's';
}

if ($role_filter !== 'all') {
    $where[] = "is_admin = ?";
    $params[] = $role_filter === 'admin' ? 1 : 0;
    $types .= 'i';
}

$whereClause = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';

// Get total count
$countSql = "SELECT COUNT(*) as total FROM users $whereClause";
$countStmt = $conn->prepare($countSql);
if (!empty($params)) {
    $countStmt->bind_param($types, ...$params);
}
$countStmt->execute();
$result = fetchStmtAssoc($countStmt);
$total = $result['total'] ?? 0;
$countStmt->close();
$totalPages = ceil($total / $limit);

// Get users
$sql = "SELECT id, username, email, full_name, is_admin, status, created_at, last_login 
        FROM users $whereClause 
        ORDER BY created_at DESC 
        LIMIT ? OFFSET ?";
$stmt = $conn->prepare($sql);
if (!empty($params)) {
    $params[] = $limit;
    $params[] = $offset;
    $types .= 'ii';
    $stmt->bind_param($types, ...$params);
} else {
    $stmt->bind_param('ii', $limit, $offset);
}
$stmt->execute();
$users = fetchAllAssoc($stmt);

$msg = $_GET['msg'] ?? '';
?>
<!DOCTYPE html>
<html lang="vi">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Quản Lý Users - Admin</title>
    <link rel="stylesheet" href="../assets/css/style.css">
    <link rel="stylesheet" href="../assets/css/admin.css">
    <link rel="preconnect" href="https://fonts.googleapis.com">
    <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
    <link href="https://fonts.googleapis.com/css2?family=Noto+Sans+SC:wght@300;400;500;700&family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
</head>
<body>
    <nav class="navbar admin-navbar">
        <div class="container">
            <div class="nav-brand">
                <h1>中文学习</h1>
                <span>Admin Dashboard</span>
            </div>
            <ul class="nav-menu">
                <li><a href="../index.php">Trang Chủ</a></li>
                <li><a href="index.php">Dashboard</a></li>
                <li><a href="users.php" class="active">Quản Lý Users</a></li>
                <li><a href="../logout.php">Đăng Xuất</a></li>
            </ul>
        </div>
    </nav>

    <main class="admin-main">
        <div class="container">
            <div class="admin-header">
                <h1>Quản Lý Users</h1>
                <p>Tổng số: <?php echo number_format($total); ?> users</p>
            </div>

            <?php if ($msg): ?>
                <div class="alert alert-success">
                    <?php
                    $messages = [
                        'banned' => 'Đã khóa user thành công',
                        'unbanned' => 'Đã mở khóa user thành công',
                        'deleted' => 'Đã xóa user thành công',
                        'admin_added' => 'Đã thêm quyền admin thành công',
                        'admin_removed' => 'Đã gỡ quyền admin thành công'
                    ];
                    echo $messages[$msg] ?? 'Thao tác thành công';
                    ?>
                </div>
            <?php endif; ?>

            <div class="admin-filters">
                <form method="GET" class="filter-form">
                    <input type="text" name="search" placeholder="Tìm kiếm username, email, tên..." value="<?php echo htmlspecialchars($search); ?>" class="filter-input">
                    <select name="status" class="filter-select">
                        <option value="all" <?php echo $status_filter === 'all' ? 'selected' : ''; ?>>Tất cả trạng thái</option>
                        <option value="active" <?php echo $status_filter === 'active' ? 'selected' : ''; ?>>Hoạt động</option>
                        <option value="inactive" <?php echo $status_filter === 'inactive' ? 'selected' : ''; ?>>Vô hiệu</option>
                        <option value="banned" <?php echo $status_filter === 'banned' ? 'selected' : ''; ?>>Bị khóa</option>
                    </select>
                    <select name="role" class="filter-select">
                        <option value="all" <?php echo $role_filter === 'all' ? 'selected' : ''; ?>>Tất cả vai trò</option>
                        <option value="admin" <?php echo $role_filter === 'admin' ? 'selected' : ''; ?>>Admin</option>
                        <option value="user" <?php echo $role_filter === 'user' ? 'selected' : ''; ?>>User</option>
                    </select>
                    <button type="submit" class="btn btn-primary">Tìm Kiếm</button>
                    <a href="users.php" class="btn btn-secondary">Reset</a>
                </form>
            </div>

            <div class="table-container">
                <table class="data-table">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Username</th>
                            <th>Email</th>
                            <th>Họ Tên</th>
                            <th>Vai Trò</th>
                            <th>Trạng Thái</th>
                            <th>Ngày Tạo</th>
                            <th>Đăng Nhập Cuối</th>
                            <th>Hành Động</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($users)): ?>
                            <tr>
                                <td colspan="9" class="text-center">Không tìm thấy user nào</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($users as $user): ?>
                                <tr>
                                    <td><?php echo $user['id']; ?></td>
                                    <td><?php echo htmlspecialchars($user['username']); ?></td>
                                    <td><?php echo htmlspecialchars($user['email']); ?></td>
                                    <td><?php echo htmlspecialchars($user['full_name'] ?? '-'); ?></td>
                                    <td>
                                        <?php if ($user['is_admin']): ?>
                                            <span class="badge badge-admin">Admin</span>
                                        <?php else: ?>
                                            <span class="badge badge-user">User</span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php
                                        $statusClass = 'badge-active';
                                        $statusText = 'Hoạt Động';
                                        if ($user['status'] === 'banned') {
                                            $statusClass = 'badge-banned';
                                            $statusText = 'Bị Khóa';
                                        } elseif ($user['status'] === 'inactive') {
                                            $statusClass = 'badge-inactive';
                                            $statusText = 'Vô Hiệu';
                                        }
                                        ?>
                                        <span class="badge <?php echo $statusClass; ?>"><?php echo $statusText; ?></span>
                                    </td>
                                    <td><?php echo date('d/m/Y', strtotime($user['created_at'])); ?></td>
                                    <td><?php echo $user['last_login'] ? date('d/m/Y H:i', strtotime($user['last_login'])) : 'Chưa đăng nhập'; ?></td>
                                    <td>
                                        <div class="action-buttons">
                                            <a href="user_detail.php?id=<?php echo $user['id']; ?>" class="btn btn-sm btn-primary">Chi Tiết</a>
                                            <?php if ($user['id'] != $_SESSION['user_id']): ?>
                                                <?php if ($user['status'] === 'banned'): ?>
                                                    <form method="POST" style="display: inline;">
                                                        <input type="hidden" name="action" value="unban">
                                                        <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                                        <button type="submit" class="btn btn-sm btn-success" onclick="return confirm('Mở khóa user này?')">Mở Khóa</button>
                                                    </form>
                                                <?php else: ?>
                                                    <form method="POST" style="display: inline;">
                                                        <input type="hidden" name="action" value="ban">
                                                        <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                                        <button type="submit" class="btn btn-sm btn-warning" onclick="return confirm('Khóa user này?')">Khóa</button>
                                                    </form>
                                                <?php endif; ?>
                                                
                                                <?php if (!$user['is_admin']): ?>
                                                    <form method="POST" style="display: inline;">
                                                        <input type="hidden" name="action" value="make_admin">
                                                        <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                                        <button type="submit" class="btn btn-sm btn-info" onclick="return confirm('Thêm quyền admin cho user này?')">Thêm Admin</button>
                                                    </form>
                                                <?php else: ?>
                                                    <form method="POST" style="display: inline;">
                                                        <input type="hidden" name="action" value="remove_admin">
                                                        <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                                        <button type="submit" class="btn btn-sm btn-secondary" onclick="return confirm('Gỡ quyền admin của user này?')">Gỡ Admin</button>
                                                    </form>
                                                <?php endif; ?>
                                                
                                                <form method="POST" style="display: inline;">
                                                    <input type="hidden" name="action" value="delete">
                                                    <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                                    <button type="submit" class="btn btn-sm btn-danger" onclick="return confirm('Xóa user này? Hành động này không thể hoàn tác!')">Xóa</button>
                                                </form>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>

            <?php if ($totalPages > 1): ?>
                <div class="pagination">
                    <?php if ($page > 1): ?>
                        <a href="?page=<?php echo $page - 1; ?>&search=<?php echo urlencode($search); ?>&status=<?php echo $status_filter; ?>&role=<?php echo $role_filter; ?>" class="btn btn-secondary">← Trước</a>
                    <?php endif; ?>
                    
                    <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                        <?php if ($i == $page || $i == 1 || $i == $totalPages || ($i >= $page - 2 && $i <= $page + 2)): ?>
                            <a href="?page=<?php echo $i; ?>&search=<?php echo urlencode($search); ?>&status=<?php echo $status_filter; ?>&role=<?php echo $role_filter; ?>" class="btn <?php echo $i == $page ? 'btn-primary' : 'btn-secondary'; ?>"><?php echo $i; ?></a>
                        <?php elseif ($i == $page - 3 || $i == $page + 3): ?>
                            <span>...</span>
                        <?php endif; ?>
                    <?php endfor; ?>
                    
                    <?php if ($page < $totalPages): ?>
                        <a href="?page=<?php echo $page + 1; ?>&search=<?php echo urlencode($search); ?>&status=<?php echo $status_filter; ?>&role=<?php echo $role_filter; ?>" class="btn btn-secondary">Sau →</a>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        </div>
    </main>

    <footer class="footer">
        <div class="container">
            <p>&copy; 2024 Học Tiếng Trung HSK. Tất cả quyền được bảo lưu.</p>
        </div>
    </footer>
</body>
</html>

